(function(window, document) {
    'use strict';
    
    const config = window.themeOptions || {};
    
    const Utils = {
        formatTime: function(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        },
        
        showXSSWarning: function(message) {
            const warningDiv = document.createElement('div');
            warningDiv.className = 'mt-3 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-700';
            warningDiv.innerHTML = `
                <div class="flex items-center">
                    <svg class="w-4 h-4 text-red-500 mr-2" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path>
                    </svg>
                    <span>${message}</span>
                </div>
            `;
            
            const oldWarning = document.querySelector('.xss-warning');
            if (oldWarning) oldWarning.remove();
            
            warningDiv.classList.add('xss-warning');
            const commentForm = document.getElementById('comment-form');
            if (commentForm) {
                commentForm.insertBefore(warningDiv, commentForm.lastElementChild);
            }
            
            setTimeout(() => {
                if (warningDiv.parentNode) warningDiv.remove();
            }, 5000);
        },
        
        testWebsiteLatency: async function(url) {
            const target = `${url}?t=${Date.now()}`;
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 4000);

            try {
                const startTime = performance.now();
                await fetch(target, {
                    mode: "no-cors",
                    cache: "no-store",
                    signal: controller.signal
                });
                const endTime = performance.now();
                clearTimeout(timeoutId);
                return {
                    accessible: true,
                    latency: Math.round(endTime - startTime)
                };
            } catch (err) {
                clearTimeout(timeoutId);
                return {
                    accessible: false,
                    latency: 3000
                };
            }
        },
        
        updateWebsiteStatus: function(el, { accessible, latency }) {
            const badge = el.querySelector('.status-badge');
            if (!badge) return;
            
            if (!accessible) {
                badge.textContent = '异常';
                badge.className = 'text-sm font-medium px-2 py-1 rounded-full bg-red-100 text-red-700 status-badge';
                el.style.pointerEvents = 'none';
                el.style.opacity = '0.7';
            } else if (latency < 1000) {
                badge.textContent = `${latency} ms`;
                badge.className = 'text-sm font-medium px-2 py-1 rounded-full bg-green-100 text-green-700 status-badge';
                el.style.pointerEvents = 'auto';
                el.style.opacity = '1';
            } else if (latency < 2000) {
                badge.textContent = `${latency} ms（缓慢）`;
                badge.className = 'text-sm font-medium px-2 py-1 rounded-full bg-yellow-100 text-yellow-700 status-badge';
                el.style.pointerEvents = 'auto';
                el.style.opacity = '0.9';
            } else {
                badge.textContent = '异常';
                badge.className = 'text-sm font-medium px-2 py-1 rounded-full bg-red-100 text-red-700 status-badge';
                el.style.pointerEvents = 'none';
                el.style.opacity = '0.7';
            }
        },
        
        autoTestAllWebsites: async function() {
            const items = document.querySelectorAll('.website-item');
            if (items.length === 0) return;
            
            items.forEach(el => {
                const badge = el.querySelector('.status-badge');
                if (badge) {
                    badge.textContent = '检测中...';
                    badge.className = 'text-sm font-medium px-2 py-1 rounded-full bg-gray-100 text-gray-600 status-badge';
                }
            });

            const results = await Promise.all(
                Array.from(items).map(async el => {
                    const url = el.dataset.url;
                    const result = await this.testWebsiteLatency(url);
                    return { el, result };
                })
            );

            results.forEach(({ el, result }) => {
                this.updateWebsiteStatus(el, result);
            });
        }
    };
    
    const Core = {
        backToTop: {
            init: function() {
                const btn = document.getElementById('backToTop');
                if (!btn) return;
                
                window.addEventListener('scroll', () => {
                    if (window.scrollY > 300) {
                        btn.classList.remove('opacity-0', 'invisible');
                        btn.classList.add('opacity-100', 'visible');
                    } else {
                        btn.classList.add('opacity-0', 'invisible');
                        btn.classList.remove('opacity-100', 'visible');
                    }
                });
                
                btn.addEventListener('click', () => {
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                });
            }
        },
        
        codeHighlight: {
            init: function() {
                if (typeof hljs === 'undefined') return;
                
                document.querySelectorAll('pre code').forEach((block) => {
                    hljs.highlightElement(block);
                    const language = hljs.highlightAuto(block.textContent).language;
                    if (language) {
                        block.parentElement.setAttribute('data-language', language);
                    }
                });
            }
        },
        
        lightbox: {
            init: function() {
                if (typeof GLightbox === 'undefined') return;
                
                GLightbox({
                    selector: '.glightbox',
                    touchNavigation: true,
                    loop: true,
                    openEffect: 'fade',
                    closeEffect: 'fade',
                    slideEffect: 'slide',
                    moreLength: 0,
                    zoomable: true,
                    draggable: true
                });
            }
        },
        
        videoControl: {
            currentlyPlayingVideo: null,
            
            init: function() {
                const allVideos = document.querySelectorAll('.video-player');
                
                allVideos.forEach(video => {        
                    video.addEventListener('play', () => {
                        if (this.currentlyPlayingVideo && this.currentlyPlayingVideo !== video) {                
                            this.currentlyPlayingVideo.pause();
                        }       
                        this.currentlyPlayingVideo = video;
                    });
                       
                    video.addEventListener('pause', () => {          
                        if (this.currentlyPlayingVideo === video) {
                            this.currentlyPlayingVideo = null;
                        }
                    });
                    
                    video.addEventListener('ended', () => {          
                        if (this.currentlyPlayingVideo === video) {
                            this.currentlyPlayingVideo = null;
                        }
                    });               
                    
                    video.addEventListener('abort', () => {
                        if (this.currentlyPlayingVideo === video) {
                            this.currentlyPlayingVideo = null;
                        }
                    });
                });
                
                document.addEventListener('visibilitychange', () => {
                    if (document.hidden && this.currentlyPlayingVideo) {
                        this.currentlyPlayingVideo.pause();
                        this.currentlyPlayingVideo = null;
                    }
                });
            }
        },
        
        counterAnimation: {
            init: function() {
                const nums = document.querySelectorAll('.num');
                if (nums.length === 0) return;
                
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            nums.forEach(num => {
                                const target = parseInt(num.dataset.target);
                                let current = 0;
                                const step = Math.ceil(target / 30);
                                const timer = setInterval(() => {
                                    current += step;
                                    if (current >= target) {
                                        current = target;
                                        clearInterval(timer);
                                    }
                                    num.textContent = current;
                                }, 30);
                            });
                            observer.disconnect();
                        }
                    });
                });
                
                const aboutSection = document.getElementById('about');
                if (aboutSection) observer.observe(aboutSection);
            }
        },
        
        skillBars: {
            init: function() {
                const skillBars = document.querySelectorAll('[data-rate]');
                if (skillBars.length === 0) return;
                
                const skillObserver = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            skillBars.forEach(bar => {
                                const rate = bar.getAttribute('data-rate');
                                bar.style.width = rate + '%';
                            });
                            skillObserver.disconnect();
                        }
                    });
                }, { threshold: 0.3 });
                
                const skillsSection = document.getElementById('skills');
                if (skillsSection) skillObserver.observe(skillsSection);
            }
        },
        
        imageFallback: {
            init: function() {
                document.querySelectorAll('img').forEach(img => {
                    img.addEventListener('error', function() {
                        this.src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAiIGhlaWdodD0iNTAiIHZpZXdCb3g9IjAgMCA1MCA1MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48Y2lyY2xlIGN4PSIyNSIgY3k9IjI1IiByPSIyNSIgZmlsbD0iI0U1RTdFRCIvPjxwYXRoIGQ9Ik0xNiAxOEMxNiAxNS43OTA5IDE3Ljc5MDkgMTQgMjAgMTRIMzBDMzIuMjA5MSAxNCAzNCAxNS43OTA5IDM0IDE4VjMyQzM0IDM0LjIwOTEgMzIuMjA5MSAzNiAzMCAzNkgyMEMxNy43OTA5IDM2IDE2IDM0LjIwOTEgMTYgMzJWMThaIiBmaWxsPSIjOUZBNUFEIi8+PHBhdGggZD0iTTI1IDI1QzMwLjUyMjggMjUgMzUgMjAuNTIyOCAzNSAxNUMzNSA5LjQ3NzE1IDMwLjUyMjggNSAyNSA1QzE5LjQ3NzEgNSAxNSA5LjQ3NzE1IDE1IDE1QzE1IDIwLjUyMjggMTkuNDc3MSAyNSAyNSAyNVoiIGZpbGw9IiM5RkE1QUQiLz48L3N2Zz4=';
                    });
                });
            }
        },
        
        smoothScroll: {
            init: function() {
                document.querySelectorAll('a[href^="#"]').forEach(anchor => {
                    anchor.addEventListener('click', function(e) {
                        const href = this.getAttribute('href');
                        if (href === '#') return;
                        
                        const target = document.querySelector(href);
                        if (target) {
                            e.preventDefault();
                            target.scrollIntoView({ behavior: 'smooth' });
                        }
                    });
                });
            }
        }
    };
    
    const Comments = {
        init: function() {
            this.initAvatarPreview();
            this.initFormValidation();
            this.initReplyLinks();
            this.initCancelReply();
        },
        
        initAvatarPreview: function() {
            const mailInput = document.getElementById('mail');
            const avatarPreview = document.getElementById('avatar-preview');
            const previewAvatar = document.getElementById('preview-avatar');
            
            if (!mailInput || !avatarPreview || !previewAvatar) return;
            
            mailInput.addEventListener('input', function() {
                const email = this.value.trim();
                if (email && email.includes('@') && !email.includes('<') && !email.includes('>')) {
                    const hash = email.toLowerCase().trim();
                    previewAvatar.src = 'https://weavatar.com/avatar/' + hash + '?s=32&d=identicon';
                    avatarPreview.classList.remove('hidden');
                } else {
                    avatarPreview.classList.add('hidden');
                }
            });
            
            if (mailInput.value && mailInput.value.includes('@') && !mailInput.value.includes('<') && !mailInput.value.includes('>')) {
                const hash = mailInput.value.toLowerCase().trim();
                previewAvatar.src = 'https://weavatar.com/avatar/' + hash + '?s=32&d=identicon';
                avatarPreview.classList.remove('hidden');
            }
        },
        
        initFormValidation: function() {
            const commentForm = document.getElementById('comment-form');
            if (!commentForm) return;
            
            const submitBtn = commentForm.querySelector('button[type="submit"]');
            const textarea = document.getElementById('textarea');
            
            if (textarea) {
                textarea.addEventListener('input', function() {
                    const value = this.value;
                    if (value.includes('<') || value.includes('>') || value.includes('javascript:')) {
                        this.classList.add('border-red-500');
                        Utils.showXSSWarning('检测到不安全字符，请移除 &lt; &gt; 或 javascript:');
                    } else {
                        this.classList.remove('border-red-500');
                        const warning = document.querySelector('.xss-warning');
                        if (warning) warning.remove();
                    }
                });
            }
            
            commentForm.addEventListener('submit', function(e) {
                const inputs = commentForm.querySelectorAll('input, textarea');
                let hasXSS = false;
                
                inputs.forEach(input => {
                    const value = input.value;
                    if (value.includes('<') || value.includes('>') || 
                        value.includes('javascript:') || 
                        value.includes('onerror=') || 
                        value.includes('onload=') ||
                        value.includes('onclick=') ||
                        value.includes('onmouseover=')) {
                        hasXSS = true;
                        input.classList.add('border-red-500');
                    } else {
                        input.classList.remove('border-red-500');
                    }
                });
                
                if (hasXSS) {
                    e.preventDefault();
                    Utils.showXSSWarning('检测到不安全的内容，请移除特殊字符。');
                    return false;
                }
                
                if (submitBtn) {
                    const originalText = submitBtn.innerHTML;
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = `
                        <svg class="animate-spin h-4 w-4 mr-2" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        提交中...
                    `;
                    
                    setTimeout(() => {
                        if (submitBtn) {
                            submitBtn.disabled = false;
                            submitBtn.innerHTML = originalText;
                        }
                    }, 5000);
                }
            });
        },
        
        initReplyLinks: function() {
            const replyLinks = document.querySelectorAll('.comment-reply-link');
            replyLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                                            
                    const cancelDiv = document.querySelector('.cancel-comment-reply');
                    if (cancelDiv) cancelDiv.classList.remove('hidden');
                    
                    const commentForm = document.getElementById('comment-form');
                    if (commentForm) {
                        commentForm.scrollIntoView({ behavior: 'smooth' });
                        
                        const author = this.getAttribute('data-author');
                        const textarea = commentForm.querySelector('textarea[name="text"]');
                        if (textarea) {
                            const currentText = textarea.value.trim();
                            const replyPrefix = '@' + author + ' ';
                            
                            if (!currentText.startsWith(replyPrefix)) {
                                textarea.value = replyPrefix + (currentText ? '\n' + currentText : '');
                            }
                            textarea.focus();
                        }
                    }
                });
            });
        },
        
        initCancelReply: function() {
            const cancelReplyBtn = document.querySelector('.cancel-comment-reply a');
            if (!cancelReplyBtn) return;
            
            cancelReplyBtn.addEventListener('click', function() {
                const cancelDiv = document.querySelector('.cancel-comment-reply');
                if (cancelDiv) cancelDiv.classList.add('hidden');

                const textarea = document.getElementById('textarea');
                if (textarea) {
                    const value = textarea.value.trim();
                    if (value.startsWith('@')) {
                        const lines = value.split('\n');
                        if (lines[0].startsWith('@')) {
                            lines.shift();
                            textarea.value = lines.join('\n').trim();
                        }
                    }
                }
            });
        }
    };
    
    const FunTools = (function() {
        let fishStartTime = null;
        let fishTimer = null;
        let audio = null;
        
        function updateClock() {
            const now = new Date();
            const hours = String(now.getHours()).padStart(2, '0');
            const minutes = String(now.getMinutes()).padStart(2, '0');
            const seconds = String(now.getSeconds()).padStart(2, '0');
            
            const currentTimeEl = document.getElementById('currentTime');
            if (currentTimeEl) {
                currentTimeEl.textContent = `${hours}:${minutes}:${seconds}`;
            }

            if (fishStartTime) {
                const diff = now - fishStartTime;
                const h = Math.floor(diff / 3600000);
                const m = Math.floor((diff % 3600000) / 60000);
                const s = Math.floor((diff % 60000) / 1000);
                
                const fishTimeEl = document.getElementById('fishTime');
                if (fishTimeEl) {
                    fishTimeEl.textContent = `${h}h ${m}m ${s}s`;
                }
            }
        }
        
        function initClock() {
            setInterval(updateClock, 1000);
            updateClock();
        }
        
        function initFishCalculator() {
            const startFishBtn = document.getElementById('startFishBtn');
            const startFishTimeEl = document.getElementById('startFishTime');
            
            if (!startFishBtn) return;
            
            startFishBtn.addEventListener('click', () => {
                if (!fishStartTime) {
                    fishStartTime = new Date();
                    const h = String(fishStartTime.getHours()).padStart(2, '0');
                    const m = String(fishStartTime.getMinutes()).padStart(2, '0');
                    const s = String(fishStartTime.getSeconds()).padStart(2, '0');
                    
                    if (startFishTimeEl) {
                        startFishTimeEl.textContent = `${h}:${m}:${s}`;
                    }
                    startFishBtn.textContent = '停止摸鱼';
                    fishTimer = setInterval(updateClock, 1000);
                } else {
                    clearInterval(fishTimer);
                    fishStartTime = null;
                    if (startFishTimeEl) {
                        startFishTimeEl.textContent = '--:--:--';
                    }
                    const fishTimeEl = document.getElementById('fishTime');
                    if (fishTimeEl) {
                        fishTimeEl.textContent = '0h 0m 0s';
                    }
                    startFishBtn.textContent = '开始摸鱼';
                }
            });
        }
        
        function loadHitokoto() {
            const hitokotoEl = document.getElementById('hitokoto');
            if (!hitokotoEl) return;
            
            hitokotoEl.textContent = '正在加载一言...';
            
            const apiUrl = config.hitokotoApi || "https://v1.hitokoto.cn/";
            fetch(apiUrl)
                .then(response => response.json())
                .then(data => {
                    hitokotoEl.textContent = data.hitokoto || '一言加载失败，请刷新页面。';
                })
                .catch(e => {
                    hitokotoEl.textContent = '一言加载失败，请检查网络。';
                });
        }
        
        function seekAudio(value) {
            if (!audio || !audio.duration) return;
            const seekTo = (value / 100) * audio.duration;
            audio.currentTime = seekTo;
        }
        
        function initAudioPlayer() {
            if (!config.musicUrl) return;
            
            audio = new Audio(config.musicUrl);
            
            audio.addEventListener('timeupdate', updateProgressBar);
            audio.addEventListener('loadedmetadata', initProgressBar);
            
            audio.addEventListener('ended', () => {
                const nowPlaying = document.getElementById('nowPlaying');
                if (nowPlaying) nowPlaying.textContent = '播放结束';
                const currentTimeSpan = document.getElementById('musicCurrentTime');
                if (currentTimeSpan) currentTimeSpan.textContent = '00:00';
                const progressBar = document.getElementById('progressBar');
                if (progressBar) progressBar.style.width = '0%';
                const progressSlider = document.getElementById('progressSlider');
                if (progressSlider) progressSlider.value = 0;
                const playIcon = document.querySelector('.play-icon');
                if (playIcon) {
                    playIcon.classList.remove('fa-pause');
                    playIcon.classList.add('fa-play');
                }
            });
            
            const playBtn = document.getElementById('playMusic');
            if (playBtn) {
                playBtn.addEventListener('click', playAudio);
            }
            
            const progressSlider = document.getElementById('progressSlider');
            if (progressSlider) {
                progressSlider.addEventListener('input', function() {
                    seekAudio(this.value);
                });
            }
        }
        
        function updateProgressBar() {
            if (!audio) return;
            
            const progressBar = document.getElementById('progressBar');
            const currentTimeSpan = document.getElementById('musicCurrentTime');
            const progressSlider = document.getElementById('progressSlider');
            
            if (audio.duration > 0) {
                const progress = (audio.currentTime / audio.duration) * 100;
                if (progressBar) progressBar.style.width = progress + '%';
                if (currentTimeSpan) currentTimeSpan.textContent = Utils.formatTime(audio.currentTime);
                if (progressSlider) progressSlider.value = progress;
            }
        }
        
        function initProgressBar() {
            if (!audio) return;
            
            const progressSlider = document.getElementById('progressSlider');
            const durationSpan = document.getElementById('musicDuration');
            
            if (progressSlider) {
                progressSlider.max = 100;
                progressSlider.value = 0;
            }
            
            if (durationSpan) {
                durationSpan.textContent = Utils.formatTime(audio.duration);
            }
        }
        
        function playAudio() {
            if (!audio) return;
            
            const playBtn = document.getElementById('playMusic');
            const playIcon = document.querySelector('.play-icon');
            
            if (audio.paused) {
                audio.play().then(() => {
                    const nowPlaying = document.getElementById('nowPlaying');
                    if (nowPlaying) nowPlaying.textContent = '正在播放音乐';
                    
                    if (playIcon) {
                        playIcon.classList.remove('fa-play');
                        playIcon.classList.add('fa-pause');
                    }
                }).catch(e => {
                    const nowPlaying = document.getElementById('nowPlaying');
                    if (nowPlaying) nowPlaying.textContent = '播放失败：请检查音乐链接';
                });
            } else {
                audio.pause();
                const nowPlaying = document.getElementById('nowPlaying');
                if (nowPlaying) nowPlaying.textContent = '已暂停';
                
                if (playIcon) {
                    playIcon.classList.remove('fa-pause');
                    playIcon.classList.add('fa-play');
                }
            }
        }
        
        function testRealSpeed() {
            const btn = document.getElementById('testBtn');
            const resultSpan = document.getElementById('speedResult');
            const speedBar = document.getElementById('speedBar');
            
            if (!btn || !resultSpan || !speedBar) return;
            
            if (btn.disabled) return;
            btn.disabled = true;
            btn.innerHTML = '<i class="fa fa-spinner fa-spin mr-2"></i> 测速中...';
            resultSpan.textContent = '--';
            speedBar.style.width = '0%';
            speedBar.style.backgroundColor = '#4F46E5';

            const startTime = performance.now();
            
            fetch('https://picsum.photos/1', {
                method: 'GET',
                cache: 'no-cache',
                mode: 'no-cors'
            })
            .then(() => {
                const endTime = performance.now();
                const ping = Math.round(endTime - startTime);

                resultSpan.textContent = ping;
                const percent = Math.min(ping / 10, 100);
                speedBar.style.width = `${percent}%`;
                
                if (ping > 500) {
                    speedBar.style.backgroundColor = '#ef4444';
                } else if (ping > 200) {
                    speedBar.style.backgroundColor = '#f59e0b';
                }
            })
            .catch(error => {
                resultSpan.textContent = '失败';
                speedBar.style.width = '100%';
                speedBar.style.backgroundColor = '#ef4444';
            })
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="fa fa-refresh mr-2"></i> 开始测速';
            });
        }
        
        function initSpeedTest() {
            const testBtn = document.getElementById('testBtn');
            if (!testBtn) return;
            
            testBtn.addEventListener('click', testRealSpeed);
        }
        
        function initHitokoto() {
            loadHitokoto();
            const refreshBtn = document.getElementById('refreshHitokoto');
            if (refreshBtn) {
                refreshBtn.addEventListener('click', loadHitokoto);
            }
        }
        
        return {
            init: function() {
                if (!config.enableFunTools) return;
                
                initClock();
                initFishCalculator();
                initHitokoto();
                initAudioPlayer();
                initSpeedTest();
            }
        };
    })();
    
    const QRCodeModal = {
        init: function() {
            const modal = document.getElementById('qrcodeModal');
            const btns = document.querySelectorAll('.sponsor-btn');
            const titleEl = document.getElementById('qrcodeTitle');
            const imgEl = document.getElementById('qrcodeImg');
            
            if (!modal || !btns.length) return;

            btns.forEach(btn => {
                btn.addEventListener('click', () => {
                    const type = btn.dataset.type;
                    let title = '收款码';
                    let imgUrl = '';
                    
                    switch(type) {
                        case 'qq':
                            title = 'QQ 收款码';
                            imgUrl = config.qqPayQr || '';
                            break;
                        case 'wechat':
                            title = '微信收款码';
                            imgUrl = config.wechatQr || '';
                            break;
                        case 'alipay':
                            title = '支付宝收款码';
                            imgUrl = config.alipayQr || '';
                            break;
                    }
                    
                    if (imgUrl) {
                        titleEl.textContent = title;
                        imgEl.src = imgUrl;
                        modal.classList.remove('hidden');
                        document.body.style.overflow = 'hidden';
                    }
                });
            });

            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.add('hidden');
                    document.body.style.overflow = '';
                }
            });
        }
    };
    
    document.addEventListener('DOMContentLoaded', function() {
        Core.backToTop.init();
        Core.codeHighlight.init();
        Core.lightbox.init();
        Core.videoControl.init();
        Core.counterAnimation.init();
        Core.skillBars.init();
        Core.imageFallback.init();
        Core.smoothScroll.init();
        
        Comments.init();
        
        FunTools.init();
        
        QRCodeModal.init();
        
        if (config.websites && config.websites.length > 0) {
            setTimeout(() => {
                Utils.autoTestAllWebsites();
            }, 2000);
        }
    });
    
    window.MyTheme = {
        Utils: Utils,
        Core: Core,
        Comments: Comments,
        FunTools: FunTools,
        QRCodeModal: QRCodeModal
    };
    
})(window, document);